const express = require('express');
const router = express.Router();
const { PrismaClient } = require('@prisma/client');
const authMiddleware = require('../middleware/auth');

const prisma = new PrismaClient();

// Protect all routes
router.use(authMiddleware);

// Get all users (admin only)
router.get('/', async (req, res) => {
  try {
    // Check if requester is admin
    if (req.user.role !== 'admin') {
      return res.status(403).json({ error: 'Admin access required' });
    }

    const users = await prisma.user.findMany({
      select: {
        id: true,
        email: true,
        name: true,
        role: true,
        email_verified: true,
        createdAt: true,
        avatarUrl: true, // Changed from avatar to avatarUrl to match schema
        vendorPortfolio: {
          select: {
            serviceTypes: true, // Changed from services to serviceTypes
            portfolioImages: true,
            // Remove reviewCount and verified as they don't exist in schema
          }
        },
        customerPreferences: {
          select: {
            savedVendors: true
          }
        }
      }
    });

    // Transform the data to match frontend expectations
    const transformedUsers = users.map(user => ({
      ...user,
      avatar: user.avatarUrl, // Map avatarUrl to avatar for frontend
      services: user.vendorPortfolio?.serviceTypes || [], // Map serviceTypes to services
      reviewCount: 0, // Default value since we don't have this in schema yet
      verified: user.email_verified, // Use email_verified as verified status
      savedProviders: user.customerPreferences?.savedVendors || []
    }));

    res.json(transformedUsers);
  } catch (error) {
    console.error('Error fetching users:', error);
    res.status(500).json({ error: 'Failed to fetch users' });
  }
});

// Verify vendor (admin only)
router.post('/:id/verify', async (req, res) => {
  try {
    // Check if requester is admin
    if (req.user.role !== 'admin') {
      return res.status(403).json({ error: 'Admin access required' });
    }

    const { id } = req.params;

    const vendor = await prisma.user.update({
      where: { id },
      data: {
        vendorPortfolio: {
          update: {
            verified: true
          }
        }
      }
    });

    res.json(vendor);
  } catch (error) {
    console.error('Error verifying vendor:', error);
    res.status(500).json({ error: 'Failed to verify vendor' });
  }
});

module.exports = router;